//////////
//
//	File:		ComApplication.c
//
//	Contains:	Application-specific code for scripting application.
//				This file is used for ***BOTH*** MacOS and Windows.
//
//	Written by:	Tim Monroe
//				Based on the MovieShell code written by Apple DTS.
//
//	Copyright:	 1994-1997 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <20>	 	10/27/97	rtm		began adding support for QuickTime video effects
//	   <19>	 	10/23/97	rtm		moved InitializeQTVR to InitApplication, TerminateQTVR to StopApplication
//	   <18>	 	10/13/97	rtm		reworked HandleApplicationMenu to use menu item identifiers
//	   <17>	 	10/10/97	rtm		StopApplication now executes quitting-time commands for all open document windows;
//									also, added phases for StopApplication
//	   <16>	 	09/24/97	rtm		upgraded to latest QuickTime VR headers
//	   <15>	 	09/18/97	rtm		added window-walking logic to DoApplicationEventLoopAction
//	   <14>	 	09/11/97	rtm		merged MacApplication.c and WinApplication.c into ComApplication.c
//	   <13>	 	08/21/97	rtm		first file for Windows; based on MacApplication.c for Mac sample code
//	   <12>	 	07/23/97	rtm		moved VRScript_CheckForTimedCommands back to DoIdle
//	   <11>	 	07/17/97	rtm		reworked DoIdle; added call to VR3DObjects_DoIdle
//	   <10>	 	07/15/97	rtm		moved VRScript_CheckForTimedCommands call from DoIdle to DoApplicationEventLoopAction
//	   <9>	 	07/14/97	rtm		fixed bug in DoIdle: put timer variable updates before _Check calls
//	   <8>	 	07/11/97	rtm		added DoApplicationEventLoopAction
//	   <7>	 	06/12/97	rtm		script file now can specify the movie file
//	   <6>	 	06/11/97	rtm		added handlers for core Apple events
//	   <5>	 	12/11/96	rtm		begun QuickTime movie work
//	   <4>	 	12/05/96	rtm		added hooks into MacFramework.c: StopApplication, InitApplicationWindowObject
//	   <3>	 	12/02/96	rtm		added cursor updating to DoIdle
//	   <2>	 	11/27/96	rtm		conversion to personal coding style; added preliminary QTVR support
//	   <1>	 	12/21/94	khs		first file
//
//////////

// header files
#include "ComApplication.h"

// global variables for Macintosh code
#if TARGET_OS_MAC
Str255					gAppName = "\pVRScript";				// the name of this application
AEEventHandlerUPP		gHandleOpenAppAEUPP;					// UPPs for our Apple event handlers
AEEventHandlerUPP		gHandleOpenDocAEUPP;
AEEventHandlerUPP		gHandlePrintDocAEUPP;
AEEventHandlerUPP		gHandleQuitAppAEUPP;
#endif

// global variables for Windows code
#if TARGET_OS_WIN32
extern HWND				ghWnd;									// the MDI frame window; this window has the menu bar
extern int				gNumWindowsOpen;
extern LPSTR			gCmdLine;
#endif

// global variables for common code
long					gMaxMilliSecToUse = 0L;		
Boolean					gQTVRMgrIsPresent = false;				// is the QuickTime VR Manager available?		
long					gQTVRMgrVersion = 0L;					// the version of the QuickTime VR Manager (currently unused)
Boolean					gHasSoundSprockets = false;				// is SoundSprockets available?
long					gAbsoluteStartingTime = 0L;				// starting time, in ticks
long					gAbsoluteElapsedTime = 0L;				// total elapsed time, in ticks
long					gNodeStartingTime = 0L;					// node starting time, in ticks
long					gNodeElapsedTime = 0L;					// node elapsed time, in ticks

Boolean					gHasQuickDraw3D = false;				// is QuickDraw 3D available?
Boolean					gHasQuickDraw3D15 = false;				// is QuickDraw 3D version 1.5 (or greater) available?

Boolean					gHasQTVideoEffects = false;				// are the QuickTime video effects available?

// external variables
extern Boolean			gAppInForeground;						// is our application in the foreground?	
extern char				gScriptFileName[kMaxFileNameLength];	// the name of the script file


//////////
//
// InitApplication
// Do any application-specific initialization.
//
// The theStartPhase parameter determines which "phase" of application start-up is executed,
// *before* the MDI frame window is created or *after*. This distinction is relevant only on
// Windows, so on MacOS, you should always use kInitAppPhase_BothPhases.
//
//////////

void InitApplication (UInt32 theStartPhase)
{
	// ***do any start-up activities that should occur before the MDI frame window is created
	if (theStartPhase & kInitAppPhase_BeforeCreateFrameWindow) {

		// see if the QuickTime video effects are available
		gHasQTVideoEffects = QTUtils_HasQuickTimeVideoEffects();
	
		// make sure that the QuickTime VR Manager is available in the present operating environment;
		// if it is, get its version and initialize it
		if (QTVRUtils_IsQTVRMgrInstalled()) {
			gQTVRMgrIsPresent = true;
			gQTVRMgrVersion = QTVRUtils_GetQTVRVersion();		// get the version of the QuickTime VR Manager
		
#if TARGET_OS_WIN32
			// initialize the QuickTime VR Manager
			InitializeQTVR();
#endif
		}
	
#if TARGET_OS_MAC
		// make sure that the Apple Event Manager is available; install handlers for required Apple events
		InstallAppleEventHandlers();
#endif

		// initialize sound capabilities
		VRSound_Init();

#if QD3D_AVAIL
		// initialize 3D object capabilities
		VR3DObjects_Init();
#endif
	
		// get the application-launch starting time 
		gAbsoluteStartingTime = TickCount();
		
	}	// end of kInitAppPhase_BeforeCreateFrameWindow

	// ***do any start-up activities that should occur after the MDI frame window is created
	if (theStartPhase & kInitAppPhase_AfterCreateFrameWindow) {
		
#if TARGET_OS_WIN32
		// on Windows, open as movie documents any files specified on the command line
		DoOpenCommandLineMovies(gCmdLine);									
#endif
	}	// end of kInitAppPhase_AfterCreateFrameWindow

}


//////////
//
// StopApplication
// Do any application-specific shut-down.
//
// The theStopPhase parameter determines which "phase" of application shut-down is executed,
// *before* any open movie windows are destroyed or *after*.
//
//////////

void StopApplication (UInt32 theStopPhase)
{
	WindowReference			myWindow;
	WindowObject			myWindowObject;
	
	// do any shut-down activities that should occur before the movie windows are destroyed
	if (theStopPhase & kStopAppPhase_BeforeDestroyWindows) {

		// execute any quitting-time commands for all open document windows
		myWindow = GetFrontMovieWindow();
		while (myWindow != NULL) {
		
			myWindowObject = GetWindowObjectFromWindow(myWindow);
			if (myWindowObject != NULL) {
				ApplicationDataHdl		myAppData;
			
				myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(myWindowObject);
				if (myAppData != NULL) {
					VRScriptAtQuitPtr	myPointer;
					
					myPointer = (VRScriptAtQuitPtr)(**myAppData).fListArray[kVREntry_QuitCommand];
					while (myPointer != NULL) {
						VRScript_ProcessScriptCommandLine(myWindowObject, myPointer->fCommandLine);
						myPointer = myPointer->fNextEntry;
					}
				}
			}
			
			myWindow = GetNextMovieWindow(myWindow);
		}
	
	}	// end of kStopAppPhase_BeforeDestroyWindows
	
	// do any shut-down activities that should occur after the movie windows are destroyed
	if (theStopPhase & kStopAppPhase_AfterDestroyWindows) {
	
		// shut down sound processing
		VRSound_Stop();

#if QD3D_AVAIL
		// shut down 3D object capabilities, if enabled
		VR3DObjects_Stop();
#endif

#if TARGET_OS_MAC
		// dispose of routine descriptors for Apple event handlers
		DisposeRoutineDescriptor(gHandleOpenAppAEUPP);
		DisposeRoutineDescriptor(gHandleOpenDocAEUPP);
		DisposeRoutineDescriptor(gHandlePrintDocAEUPP);
		DisposeRoutineDescriptor(gHandleQuitAppAEUPP);
#endif
	
#if TARGET_OS_WIN32
		// terminate QuickTime VR Manager
		if (gQTVRMgrIsPresent)
			TerminateQTVR();
#endif
			
	}	// end of kStopAppPhase_AfterDestroyWindows
	
}


//////////
//
// DoIdle
// Do any processing that can/should occur at idle time.
//
//////////

void DoIdle (WindowReference theWindow)
{
	WindowObject 		myWindowObject = NULL;
	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	// update our timer global variables; this also happens in DoApplicationEventLoopAction
	gAbsoluteElapsedTime = TickCount() - gAbsoluteStartingTime; 
	gNodeElapsedTime = TickCount() - gNodeStartingTime;
	
	myWindowObject = GetWindowObjectFromWindow(theWindow);
	if (myWindowObject != NULL) {
		MovieController		myMC = NULL;
		ApplicationDataHdl	myAppData = NULL;
		Boolean				myNeedUpdate = false;
	
		myMC = (**myWindowObject).fController;
		if (myMC != NULL) {

#if TARGET_OS_MAC
			// restore the cursor to the arrow
			// if it's outside the front movie window or outside the window's visible region
			if (theWindow == GetFrontMovieWindow()) {
				Rect	myRect;
				Point	myPoint;
				
				GetMouse(&myPoint);
				MCGetControllerBoundsRect(myMC, &myRect);
				if (!MacPtInRect(myPoint, &myRect) || !PtInRgn(myPoint, GetPortFromWindowReference(theWindow)->visRgn))
					MacSetCursor(&qd.arrow);
			}
#endif
		}
		
		// see whether we need to execute any time-triggered commands
		VRScript_CheckForTimedCommands(myWindowObject);

		// see whether we need to execute any angle-triggered commands
		VRScript_CheckForAngleCommands(myWindowObject);

		// if any view parameters have changed, we need to update window		
		myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(myWindowObject);
		if (myAppData != NULL)
			if ((**myAppData).fViewHasChanged || (**myAppData).fSoundHasChanged) 
				myNeedUpdate = true;
				
		// give any embedded QuickTime movies some processor time
		myNeedUpdate |= VRMoov_DoIdle(myWindowObject);
		
		// give any QuickTime video effects some processor time
		if (gHasQTVideoEffects)
			myNeedUpdate |= VREffects_DoIdle(myWindowObject);
		
#if QD3D_AVAIL
		// give 3D animation and texture-mapping some processor time
		if (gHasQuickDraw3D)
			myNeedUpdate |= VR3DObjects_DoIdle(myWindowObject);
#endif

		if (myNeedUpdate)
			QTVRUpdate((**myWindowObject).fInstance, kQTVRCurrentMode);
	}

	MacSetPort(mySavedPort);
}


//////////
//
// DoUpdateWindow
// Update the specified window.
//
//////////

void DoUpdateWindow (WindowReference theWindow, Rect *theRefreshArea)
{
#pragma unused(theRefreshArea)
	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	BeginUpdate(theWindow);
	
	// draw the movie controller and its movie
	MCDoAction(GetMCFromWindow(theWindow), mcActionDraw, theWindow);
	
	EndUpdate(theWindow);
	MacSetPort(mySavedPort);
}


//////////
//
// HandleContentClick
// Handle mouse button clicks in the specified window.
//
//////////

void HandleContentClick (WindowReference theWindow, EventRecord *theEvent)
{
#pragma unused(theEvent)

	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	// @@@INSERT APPLICATION-SPECIFIC CONTENT CLICKING FUNCTIONALITY HERE

	MacSetPort(mySavedPort);
}


//////////
//
// HandleApplicationKeyPress
// Handle application-specific key presses.
// Returns true if the key press was handled, false otherwise.
//
//////////

Boolean HandleApplicationKeyPress (char theCharCode)
{
	Boolean			isHandled = true;	// assume we'll handle the key press
	QTVRInstance	myInstance;
	
	// make sure we can safely call the QTVR API
	if (!gQTVRMgrIsPresent)
		return(false);
	
	myInstance = GetQTVRInstanceFromFrontWindow();
	if (myInstance == NULL)
		return(false);
	
	switch (theCharCode) {
		// take keypad keys as a nudge in the appropriate direction
		case '1':	QTVRNudge(myInstance, kQTVRDownLeft);		break;
		case '2':	QTVRNudge(myInstance, kQTVRDown);			break;
		case '3':	QTVRNudge(myInstance, kQTVRDownRight);		break;
		case '4':	QTVRNudge(myInstance, kQTVRLeft);			break;
		case '6':	QTVRNudge(myInstance, kQTVRRight);			break;
		case '7':	QTVRNudge(myInstance, kQTVRUpLeft);			break;
		case '8':	QTVRNudge(myInstance, kQTVRUp);				break;
		case '9':	QTVRNudge(myInstance, kQTVRUpRight);		break;	
		default:
			isHandled = false;
			break;
	}

	if (isHandled)
		QTVRUpdate(myInstance, kQTVRCurrentMode);
		
	return(isHandled);
}


#if TARGET_OS_MAC
//////////
//
// CreateMovieWindow
// Create a window to display a movie in.
//
//////////

WindowRef CreateMovieWindow (Rect *theRect, Str255 theTitle)
{
	WindowRef			myWindow;
		
	myWindow = NewCWindow(NULL, theRect, theTitle, false, noGrowDocProc, (WindowPtr)-1L, true, 0);
	return(myWindow);
}
#endif


//////////
//
// HandleApplicationMenu
// Handle selections in the application's menus.
//
// The theMenuItem parameter is a UInt16 version of the Windows "menu item identifier". 
// When called from Windows, theMenuItem is simply the menu item identifier passed to the window proc.
// When called from MacOS, theMenuItem is constructed like this:
// 	*high-order 8 bits == the Macintosh menu ID (1 thru 256)
// 	*low-order 8 bits == the Macintosh menu item (sequential from 1 to ordinal of last menu item in menu)
// In this way, we can simplify the menu-handling code. There are, however, some limitations,
// mainly that the menu item identifiers on Windows must be derived from the Mac values. 
//
//////////

void HandleApplicationMenu (UInt16 theMenuItem)
{
#pragma unused(theMenuItem)
	// currently, we don't have any application-specific menus to handle
}


//////////
//
// AdjustApplicationMenus
// Adjust state of items in the application's menus.
//
//////////

void AdjustApplicationMenus (WindowReference theWindow, MenuReference theMenu)
{
#pragma unused(theWindow, theMenu)
	// currently, we don't have any application-specific menus to adjust
}


//////////
//
// DoApplicationEventLoopAction
// Perform any application-specific event loop actions.
//
// Return true to indicate that we've completely handled the event here, false otherwise.
//
//////////

Boolean DoApplicationEventLoopAction (EventRecord *theEvent)
{
#pragma unused(theEvent)
	
	WindowReference		myWindow;
	WindowObject		myWindowObject;

	// update our timer global variables; this also happens in DoIdle
	gAbsoluteElapsedTime = TickCount() - gAbsoluteStartingTime; 
	gNodeElapsedTime = TickCount() - gNodeStartingTime;
	 
	// check for completed sounds and movies in all open movie windows
	myWindow = GetFrontMovieWindow();

	while (myWindow != NULL) {
		myWindowObject = GetWindowObjectFromWindow(myWindow);
		if (myWindowObject != NULL) {
			// check for completed sounds and movies
			VRSound_CheckForCompletedSounds(myWindowObject);
			VRMoov_CheckForCompletedMovies(myWindowObject);
		}

		myWindow = GetNextMovieWindow(myWindow);

	}	// while
	
	return(false);		// we didn't intercept the event
}


//////////
//
// AddControllerFunctionality
// Configure the movie controller.
//
//////////

void AddControllerFunctionality (MovieController theMC)
{
	long			myControllerFlags;
	
	// CLUT table use	
	MCDoAction(theMC, mcActionGetFlags, &myControllerFlags);
	MCDoAction(theMC, mcActionSetFlags, (void *)(myControllerFlags | mcFlagsUseWindowPalette));

	// enable keyboard event handling	
	MCDoAction(theMC, mcActionSetKeysEnabled, (void *)true);
	
	// disable drag support
	MCDoAction(theMC, mcActionSetDragEnabled, (void *)false);
}


//////////
//
// InitApplicationWindowObject
// Do any application-specific initialization of the window object.
//
//////////

void InitApplicationWindowObject (WindowObject theWindowObject)
{
	Track					myQTVRTrack = NULL;
	Movie					myMovie = NULL;
	MovieController			myMC = NULL;
	QTVRInstance			myInstance = NULL;
	ApplicationDataHdl		myAppData;
		
	if (theWindowObject == NULL)
		return;

	// make sure we can safely call the QTVR API
	if (!gQTVRMgrIsPresent)
		return;

	// find the QTVR track, if there is one
	myMC = (**theWindowObject).fController;
	myMovie = (**theWindowObject).fMovie;
	myQTVRTrack = QTVRGetQTVRTrack(myMovie, 1);
	
	QTVRGetQTVRInstance(&myInstance, myQTVRTrack, myMC);
	(**theWindowObject).fInstance = myInstance;

	// do any application-specific window configuration
	if (myInstance != NULL) {
		
		// set unit to radians
		QTVRSetAngularUnits(myInstance, kQTVRRadians);

		// do window configuration
		myAppData = (ApplicationDataHdl)NewHandleClear(sizeof(ApplicationDataRecord));
		(**theWindowObject).fAppData = (Handle)myAppData;
		
		if (myAppData != NULL) {
			
			// initialize for sounds
			VRSound_InitWindowData(theWindowObject);

#if QD3D_AVAIL
			// initialize for 3D objects
			if (gHasQuickDraw3D)
				VR3DObjects_InitWindowData(theWindowObject);
#endif
			// initialize for QuickTime video effects
			if (gHasQTVideoEffects)
				VREffects_InitWindowData(theWindowObject);

			// allocate a routine descriptor for our prescreen routine						
			(**myAppData).fPrescreenProc = NewQTVRImagingCompleteProc(VRScript_PrescreenRoutine);	

			// install an intercept procedure and a prescreen procedure
			VRScript_InstallInterceptRoutine(myInstance, theWindowObject);
			VRScript_InstallPrescreenRoutine(myInstance, theWindowObject);
			
			// install node-entering and node-leaving procedures
			QTVRSetEnteringNodeProc(myInstance, NewQTVREnteringNodeProc(VRScript_EnteringNodeProc), (SInt32)theWindowObject, 0);
			QTVRSetLeavingNodeProc(myInstance, NewQTVRLeavingNodeProc(VRScript_LeavingNodeProc), (SInt32)theWindowObject, 0);
			
			// install hot spot procedures
			VRScript_InstallHotSpotInterceptProc(myInstance, theWindowObject);
			VRScript_InstallMouseOverHotSpotProc(myInstance, theWindowObject);
				
			// read the command script;
			// this opens a script that is in the same directory as the QTVR movie file
			if (strlen(gScriptFileName) > 0)
				VRScript_OpenScriptFile(theWindowObject, gScriptFileName);
			else
#if TARGET_OS_MAC
				VRScript_OpenScriptFile(theWindowObject, "myScript");
#elif TARGET_OS_WIN32
				VRScript_OpenScriptFile(theWindowObject, "C:\\WWDC\\myScript");
#endif

			// ***TESTING***
			// this is a good place to test out commands without having to read them from a script file
			if (false) {
				char	myCmdLine1[kMaxCmdLineLength] = "PlayTransEffect	-1	-1	-1	smpt	409		0";
				char	myCmdLine2[kMaxCmdLineLength] = "PlayTransEffect	2	3	1	smpt	74		0";
				char	myCmdLine3[kMaxCmdLineLength] = "PlayTransEffect	-1	-1	-1	dslv	74		0";
				char	myCmdLine4[kMaxCmdLineLength] = "PlayTransEffect	3	2	-1	dslv	74		0";
				
				VRScript_ProcessScriptCommandLine(theWindowObject, myCmdLine1);
				//VRScript_ProcessScriptCommandLine(theWindowObject, myCmdLine2);
				//VRScript_ProcessScriptCommandLine(theWindowObject, myCmdLine3);
				//VRScript_ProcessScriptCommandLine(theWindowObject, myCmdLine4);
			}
			// ***TESTING***
			
			// initialize for first node
			(**myAppData).fViewHasChanged = true;
			(**myAppData).fSoundHasChanged = true;
			
			// node-entering procedure is NOT automatically called for the first node, so do it now....
			VRScript_EnteringNodeProc(myInstance, QTVRGetCurrentNodeID(myInstance), theWindowObject);
			
			QTVRUpdate(myInstance, kQTVRCurrentMode);
		}
	}
}


//////////
//
// RemoveApplicationWindowObject
// Do any application-specific clean-up of the window object.
//
//////////

void RemoveApplicationWindowObject (WindowObject theWindowObject)
{
	ApplicationDataHdl	myAppData;
	
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;

	VRSound_DumpWindowData(theWindowObject);
	VRMoov_DumpNodeMovies(theWindowObject);
	VRMoov_DumpSceneMovies(theWindowObject);

#if QD3D_AVAIL
	// clean-up after QuickDraw 3D
	if (gHasQuickDraw3D)
		VR3DObjects_DumpWindowData(theWindowObject);
#endif
		
	// clean-up after QuickTime video effects
	if (gHasQTVideoEffects)
		VREffects_DumpWindowData(theWindowObject);

#if TARGET_OS_MAC
	// get rid of routine descriptors
	DisposeRoutineDescriptor((**myAppData).fPrescreenProc);
	DisposeRoutineDescriptor((**myAppData).fBackBufferProc);
#endif
		
	// get rid of linked lists
	VRScript_DeleteAllLists(theWindowObject);
			
	// get rid of window data
	DisposeHandle((Handle)myAppData);
		
	// DoDestroyMovieWindow in MacFramework.c or MovieWndProc in WinFramework.c
	// releases the window object itself
}


//////////
//
// ApplicationMCActionFilterProc 
// Intercept some mc actions for the QuickTime or QuickTime VR movie controller.
//
// NOTE: The theRefCon parameter is a handle to a window object record.
//
//////////

PASCAL_RTN Boolean ApplicationMCActionFilterProc (MovieController theMC, short theAction, void *theParams, long theRefCon)
{
#pragma unused(theMC, theParams)

	Boolean				isHandled = false;
	WindowObject		myWindowObject = NULL;
	ApplicationDataHdl	myAppData;
	
	if (theMC == NULL)
		return(isHandled);
		
	myWindowObject = (WindowObject)theRefCon;
	if (myWindowObject == NULL)
		return(isHandled);
		
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(myWindowObject);
	// normally we would bail if myAppData == NULL, but we need to handle QuickTime movies here as well

	switch (theAction) {
	
		// handle window resizing
		case  mcActionControllerSizeChanged:
			SizeWindowToMovie(myWindowObject);
			
			// because the window size has changed, 
			// we need to recalculate QD3D camera's aspect ratio and resize the QD3D draw context
#if QD3D_AVAIL
			if ((**myWindowObject).fInstance != NULL)
				if (gHasQuickDraw3D) {
					VR3DObjects_SetCameraAspectRatio(myWindowObject);								 
					VR3DObjects_UpdateDrawContext(myWindowObject);								 
					QTVRUpdate((**myWindowObject).fInstance, kQTVRCurrentMode);
				}
#endif
			break;
			
		// handle idle events
		case mcActionIdle:
			DoIdle((**myWindowObject).fWindow);
			break;
			
		// handle update events
		case mcActionDraw:
			break;

		default:
			break;
			
	}	// switch (theAction)
	
	return(isHandled);	
}


#if TARGET_OS_MAC
//////////
//
// InstallAppleEventHandlers 
// Install handlers for Apple events.
//
//////////

void InstallAppleEventHandlers (void)
{
	OSErr		myErr;
	long		myAttrs;
	
	// see whether the Apple Event Manager is available in the present operating environment;
	// if it is, install handlers for the four required Apple events
	myErr = Gestalt(gestaltAppleEventsAttr, &myAttrs);
	if (myErr == noErr) {
		if (myAttrs & (1L << gestaltAppleEventsPresent)) {
			// create routine descriptors for the Apple event handlers
			gHandleOpenAppAEUPP = NewAEEventHandlerProc(HandleOpenApplicationAppleEvent);
			gHandleOpenDocAEUPP = NewAEEventHandlerProc(HandleOpenDocumentAppleEvent);
			gHandlePrintDocAEUPP = NewAEEventHandlerProc(HandlePrintDocumentAppleEvent);
			gHandleQuitAppAEUPP = NewAEEventHandlerProc(HandleQuitApplicationAppleEvent);
			
			// install the handlers
			AEInstallEventHandler(kCoreEventClass, kAEOpenApplication, gHandleOpenAppAEUPP, 0L, false);
			AEInstallEventHandler(kCoreEventClass, kAEOpenDocuments, gHandleOpenDocAEUPP, 0L, false);
			AEInstallEventHandler(kCoreEventClass, kAEPrintDocuments, gHandlePrintDocAEUPP, 0L, false);
			AEInstallEventHandler(kCoreEventClass, kAEQuitApplication, gHandleQuitAppAEUPP, 0L, false);
		}
	}
}
		
		
//////////
//
// HandleOpenApplicationAppleEvent 
// Handle the open-application Apple event.
//
//////////

PASCAL_RTN OSErr HandleOpenApplicationAppleEvent (const AppleEvent *theMessage, const AppleEvent *theReply, long theRefcon)			
{
#pragma unused(theMessage, theReply, theRefcon)

	// we don't need to do anything special when opening the application
	return(noErr);
}


//////////
//
// HandleOpenDocumentAppleEvent 
// Handle the open-document Apple event. This is based on Inside Macintosh: IAC, pp. 4-15f.
//
// Here we process an Open Documents AE only for files of type kScriptFileType and creator kScriptFileCreator.
//
//////////

PASCAL_RTN OSErr HandleOpenDocumentAppleEvent (const AppleEvent *theMessage, const AppleEvent *theReply, long theRefcon)			
{
#pragma unused(theReply, theRefcon)

	long			myIndex;
	long			myItemsInList;
	AEKeyword		myKeyWd;
	OSErr			myErr;
	OSErr			myIgnoreErr;
	AEDescList 	 	myDocList;
	long			myActualSize;
	DescType		myTypeCode;
	FSSpec			myFSSpec;
	
	// get the direct parameter and put it into myDocList
	myDocList.dataHandle = NULL;
	myErr = AEGetParamDesc(theMessage, keyDirectObject, typeAEList, &myDocList);
	
	// count the descriptor records in the list
	if (myErr == noErr)
		myErr = AECountItems(&myDocList, &myItemsInList);
	else
		myItemsInList = 0;
	
	// open each specified file
	for (myIndex = 1; myIndex <= myItemsInList; myIndex++)
		if (myErr == noErr) {
			myErr = AEGetNthPtr(&myDocList, myIndex, typeFSS, &myKeyWd, &myTypeCode, (Ptr)&myFSSpec, sizeof(myFSSpec), &myActualSize);
			if (myErr == noErr) {
				FInfo		myFinderInfo;
			
				// verify that the file type is kScriptFileType and the creator is kScriptFileCreator;
				// to do this, get the Finder information
				myErr = FSpGetFInfo(&myFSSpec, &myFinderInfo);	
				if (myErr == noErr) {
					if ((myFinderInfo.fdType == kScriptFileType) && (myFinderInfo.fdCreator == kScriptFileCreator))
						// we've got a script file; find the target QTVR movie and open it
						VRScript_FindAndOpenQTVRMovieFile(&myFSSpec);
				}
			}
		}

	if (myDocList.dataHandle)
		myIgnoreErr = AEDisposeDesc(&myDocList);
	
	// make sure we open the document in the foreground		
	gAppInForeground = true;
	return(myErr);
}


//////////
//
// HandlePrintDocumentAppleEvent 
// Handle the print-document Apple event.
// NOT YET IMPLEMENTED: because we don't want to print our scripts, just act on them.
//
//////////

PASCAL_RTN OSErr HandlePrintDocumentAppleEvent (const AppleEvent *theMessage, const AppleEvent *theReply, long theRefcon)			
{
#pragma unused(theMessage, theReply, theRefcon)

	return(errAEEventNotHandled);
}


//////////
//
// HandleQuitApplicationAppleEvent 
// Handle the quit-application Apple event.
//
//////////

PASCAL_RTN OSErr HandleQuitApplicationAppleEvent (const AppleEvent *theMessage, const AppleEvent *theReply, long theRefcon)			
{
#pragma unused(theMessage, theReply, theRefcon)

	// close down the entire framework and application
	QuitFramework();
	return(noErr);
}
#endif	// TARGET_OS_MAC

